﻿using UnityEngine;
using System.Collections;

using UnityEditor;

[CustomPropertyDrawer(typeof(Range))]
public class RangeEditor : PropertyDrawer {

	// Ta szuflada właściwości będzie mieć wysokość dwóch wierszy: 
	// w pierwszym będzie wyświetlony suwak, a w drugim pola tekstowe
	// pozwalające na bezpośrednią zmianę wartości
	const int LINE_COUNT = 2;

	public override float GetPropertyHeight (SerializedProperty property, GUIContent label)
	{
		// Zwracamy liczbę pikseli odpowiadającą wysokość tej właściwości
		return base.GetPropertyHeight (property, label) * LINE_COUNT;
	}

	public override void OnGUI (Rect position, SerializedProperty property, GUIContent label)
	{

		// Pobieramy obiekty reprezentujące pola właściwości Range
		var minProperty = property.FindPropertyRelative("min");
		var maxProperty = property.FindPropertyRelative("max");

		var minLimitProperty = property.FindPropertyRelative("minLimit");
		var maxLimitProperty = property.FindPropertyRelative("maxLimit");

		// Wszelkie kontrolki wewnątrz bloku using propertyScope będą prawidłowo
		// operować na prefabrykatach - wartości zmienione w stosunku do prefabrykatu 
		// będą wyświetlone pogrubioną czcionką, a użytkownik będzie mógł kliknąć 
		// ją prawym przyciskiem myszy i wybrać opcje przywrócenia wartości z prefabrykatu
		using (var propertyScope = new EditorGUI.PropertyScope(position, label, property)) {

			// Wyświetlamy etykietę; ta metoda zwraca obiekt Rect 
			// który może określić położenie i wielkość kontrolki obok etykiety
			Rect sliderRect = EditorGUI.PrefixLabel(position, label);

			// Tworzymy prostokąty dla poszczególnych kontrolek

			// Wyznaczamy wielkość jednego wiersza
			var lineHeight = position.height / LINE_COUNT;

			// Suwak musi mieć jeden wiersz wysokości
			sliderRect.height = lineHeight;

			// Obszar dla dwóch pól ma ten sam kształt co suwak, 
			// przesuwamy go jednak o jeden wiersz w dół
			var valuesRect = sliderRect;
			valuesRect.y += sliderRect.height;

			// Wyznaczamy prostokąty dla dwóch pól tekstowych
			var minValueRect = valuesRect;
			minValueRect.width /= 2.0f;

			var maxValueRect = valuesRect;
			maxValueRect.width /= 2.0f;
			maxValueRect.x += minValueRect.width;

			// Pobieramy wartości typu float
			var minValue = minProperty.floatValue;
			var maxValue = maxProperty.floatValue;

			// Początek sprawdzenia zmiany - robimy to by prawidłowo obsługiwać
			// edycję wielu obiektów
			EditorGUI.BeginChangeCheck();

			// Wyświetlamy suwak
			EditorGUI.MinMaxSlider(
				sliderRect,
				ref minValue,
				ref maxValue, 
				minLimitProperty.floatValue, 
				maxLimitProperty.floatValue
			);

			// Wyświetlamy pola
			minValue = EditorGUI.FloatField(minValueRect, minValue);
			maxValue = EditorGUI.FloatField(maxValueRect, maxValue);

			// Czy wartości uległy zmianie?
			var valueWasChanged = EditorGUI.EndChangeCheck();

			if (valueWasChanged) {
				// Zapisujemy zmodyfikowane wartości
				minProperty.floatValue = minValue;
				maxProperty.floatValue = maxValue;
			}
		}


	}
}
